/*
 * Copyright  2014 Daniel Taliun, Johann Gamper and Cristian Pattaro. All rights reserved.
 *
 * This file is part of S-MIG++.
 *
 * S-MIG++ is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * S-MIG++ is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with S-MIG++. If not, see <http://www.gnu.org/licenses/>.
 */

#include "include/Cell.h"

const double Cell::EPSILON = 0.000000001;

Db* Cell::db = NULL;
const char* Cell::method = NULL;
double Cell::samples = numeric_limits<double>::quiet_NaN();
unsigned long int Cell::seed = 0ul;

Cell::Cell(unsigned int segment_i, unsigned int segment_j, unsigned int start_i, unsigned int end_i, unsigned start_j, unsigned int end_j):
	segment_i(segment_i), segment_j(segment_j), start_i(start_i), end_i(end_i), start_j(start_j), end_j(end_j),
	n_sampled_noninf(0ul), n_sampled_ld(0ul), n_sampled_ehr(0ul) {

}

Cell::~Cell() {

}

void Cell::set_db(Db* db) {
	Cell::db = db;
}

void Cell::set_method(const char* method) {
	Cell::method = method;
}

void Cell::set_samples(double samples) {
	Cell::samples = samples;
}

void Cell::set_seed(unsigned long int seed) {
	Cell::seed = seed;
}

void Cell::get_coordinates(unsigned int* segment_i, unsigned int* segment_j) {
	*segment_i = this->segment_i;
	*segment_j = this->segment_j;
}

unsigned int Cell::get_segment_i() {
	return segment_i;
}

unsigned int Cell::get_segment_j() {
	return segment_j;
}

unsigned int Cell::get_start_i() {
	return start_i;
}

unsigned int Cell::get_end_i() {
	return end_i;
}

unsigned int Cell::get_start_j() {
	return start_j;
}

unsigned int Cell::get_end_j() {
	return end_j;
}

unsigned long int Cell::get_n_total() {
	if ((start_i == start_j) && (end_i == end_j)) { // on diagonal
		return (((end_i - start_i + 1u) * (end_j - start_j)) / 2u);
	} else {
		return ((end_i - start_i + 1u) * (end_j - start_j + 1u));
	}
}

unsigned long int Cell::get_n_sampled_noninf() {
	return n_sampled_noninf;
}

unsigned long int Cell::get_n_sampled_ld() {
	return n_sampled_ld;
}

unsigned long int Cell::get_n_sampled_ehr() {
	return n_sampled_ehr;
}

unsigned long int Cell::sample() throw (Exception) {
	gsl_rng* generator = NULL;
	unsigned int* source = NULL;
	unsigned int destination[] = {0u, 0u};

	unsigned int n_samples = 0u;

	unsigned int dim_i = end_i - start_i + 1u;
	unsigned int dim_j = end_j - start_j + 1u;

	unsigned int i = 0u;
	unsigned int j = 0u;

	AlgorithmCI* algorithm_ci = NULL;
	double lower_ci = 0.0;
	double upper_ci = 0.0;

	algorithm_ci = AlgorithmFactory::create(*db, method);

	generator = gsl_rng_alloc(gsl_rng_mt19937);
	if (generator == NULL) {
		throw Exception(__FILE__, __LINE__, "Error while creating pseudo random number generator.");
	}

	// use Cantor pairing function to create a unique number from two numbers
	gsl_rng_set(generator, seed + auxiliary::get_cantor_pairing_number(segment_i, segment_j));

	if ((start_i == start_j) && (end_i == end_j)) { // on diagonal
		n_samples = samples * dim_i * (dim_j - 1u) / 2u;

		if (n_samples == 0u) {
			n_samples = 1u;
		}

		source = (unsigned int*)malloc(dim_i * sizeof(unsigned int));
		if (source == NULL) {
			throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
		}

		for (unsigned int k = 0u; k < dim_i; ++k) {
			source[k] = k;
		}

		while (n_samples > 0u) {
			gsl_ran_choose(generator, destination, 2u, source, dim_i, sizeof(unsigned int));

			if (destination[0u] > destination[1u]) {
				i = destination[1u];
				j = destination[0u];
			} else {
				i = destination[0u];
				j = destination[1u];
			}

			i += start_i;
			j += start_j;

			algorithm_ci->get_CI(i, j, &lower_ci, &upper_ci);
			if (!std::isnan(lower_ci) && !std::isnan(upper_ci)) {
				if (((auxiliary::fcmp(lower_ci, 0.7, EPSILON) >= 0) && (auxiliary::fcmp(upper_ci, 0.98, EPSILON) >= 0)) ||
						((auxiliary::fcmp(lower_ci, -0.98, EPSILON) <= 0) && (auxiliary::fcmp(upper_ci, -0.7, EPSILON) <= 0))) {
					++n_sampled_ld;
				} else if ((auxiliary::fcmp(lower_ci, -0.9, EPSILON) >= 0) && (auxiliary::fcmp(upper_ci, 0.9, EPSILON) <= 0)) {
					++n_sampled_ehr;
				} else {
					++n_sampled_noninf;
				}
			} else {
				++n_sampled_noninf;
			}

			--n_samples;
		}

		free(source);
		source = NULL;
	} else {
		n_samples = samples * dim_i * dim_j;

		if (n_samples == 0u) {
			throw Exception(__FILE__, __LINE__, "Cell size is too small for the specified sampling.");
		}

		while (n_samples > 0) {
			i = start_i + gsl_rng_uniform_int(generator, dim_i);
			j = start_j + gsl_rng_uniform_int(generator, dim_j);

			algorithm_ci->get_CI(i, j, &lower_ci, &upper_ci);
			if (!std::isnan(lower_ci) && !std::isnan(upper_ci)) {
				if (((auxiliary::fcmp(lower_ci, 0.7, EPSILON) >= 0) && (auxiliary::fcmp(upper_ci, 0.98, EPSILON) >= 0)) ||
						((auxiliary::fcmp(lower_ci, -0.98, EPSILON) <= 0) && (auxiliary::fcmp(upper_ci, -0.7, EPSILON) <= 0))) {
					++n_sampled_ld;
				} else if ((auxiliary::fcmp(lower_ci, -0.9, EPSILON) >= 0) && (auxiliary::fcmp(upper_ci, 0.9, EPSILON) <= 0)) {
					++n_sampled_ehr;
				} else {
					++n_sampled_noninf;
				}
			} else {
				++n_sampled_noninf;
			}

			--n_samples;
		}
	}

	gsl_rng_free(generator);
	generator = NULL;

	delete algorithm_ci;
	algorithm_ci = NULL;

	return n_sampled_noninf + n_sampled_ld + n_sampled_ehr;
}

void Cell::write(Writer* writer) throw (Exception) {
	unsigned long int local_seed = seed + ((segment_i + segment_j) * (segment_i + segment_j + 1ul)) / 2ul + segment_j;

	writer->write("%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\n",
			segment_i, segment_j, start_i, end_i, start_j, end_j, local_seed, n_sampled_ld, n_sampled_ehr, n_sampled_noninf);
}
