/*
 * Copyright  2014 Daniel Taliun, Johann Gamper and Cristian Pattaro. All rights reserved.
 *
 * This file is part of S-MIG++.
 *
 * S-MIG++ is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * S-MIG++ is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with S-MIG++. If not, see <http://www.gnu.org/licenses/>.
 */

#include "include/ContourBuilder.h"

const double ContourBuilder::EPSILON = 0.000000001;

ContourBuilder::ContourBuilder(Db& db, const char* ci_method, const char* sci_method, double probability, double proportion, double samples, unsigned int n_segments, bool adjust, unsigned long int seed) throw (Exception) :
	db(&db), ci_method(ci_method), sci_method(sci_method), samples(samples), proportion(proportion), n_segments(n_segments), profile_factory(NULL),
	seed(seed), n_cells(0ul),
	segment_i(0u), segment_j(0u),
	n_computations(0ul), n_candidate_profiles(0ul),
	estimated_contour(NULL),
	segments(NULL), max_pool_size(0ul) {

	n_cells = ((unsigned long int)n_segments * ((unsigned long int)n_segments + 1ul)) / 2ul;

	if (adjust) {
		alpha = 1.0 - pow(probability, 1.0 / n_cells);
	} else {
		alpha = 1.0 - probability;
	}

	// for one-sided CI we use 2 * alpha
	alpha = 2.0 * alpha;

	estimated_contour = (unsigned int*)malloc(db.get_n_markers() * sizeof(unsigned int));
	if (estimated_contour == NULL) {
		throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
	}

	for (unsigned int i = 0u; i < db.get_n_markers(); ++i) {
		estimated_contour[i] = i;
	}

	if (seed == 0ul) {
		gsl_rng* generator = gsl_rng_alloc(gsl_rng_ranlxs2);
		if (generator == NULL) {
			throw Exception(__FILE__, __LINE__, "Error while creating pseudo random number generator.");
		}

		gsl_rng_set(generator, time(NULL));
		this->seed = gsl_rng_get(generator);

		gsl_rng_free(generator);
		generator = NULL;
	}

	generate_segments();

	if (auxiliary::strcmp_ignore_case(sci_method, ProfileFactory::FS) == 0) {
		profile_factory = new ProfileFactory_FS();
	} else {
		throw Exception(__FILE__, __LINE__, "Unrecognized simultaneous CI method.");
	}
}


ContourBuilder::~ContourBuilder() {
	if (profile_factory != NULL) {
		free(profile_factory);
		profile_factory = NULL;
	}

	if (segments != NULL) {
		free(segments);
		segments = NULL;
	}

	if (estimated_contour != NULL) {
		free(estimated_contour);
		estimated_contour = NULL;
	}
}

unsigned int ContourBuilder::get_n_segments() {
	return n_segments;
}

unsigned int ContourBuilder::get_n_cells() {
	return n_cells;
}

unsigned long int ContourBuilder::get_seed() {
	return seed;
}

double ContourBuilder::get_alpha() {
	return alpha;
}

unsigned long int ContourBuilder::get_n_computations() {
	return n_computations;
}

unsigned long int ContourBuilder::get_n_candidate_profiles() {
	return n_candidate_profiles;
}

unsigned long int ContourBuilder::get_estimated_contour_area() {
	unsigned long int estimated_contour_area = 0ul;

	for (unsigned int i = 1u; i < db->get_n_markers(); ++i) {
		estimated_contour_area += i - estimated_contour[i];
	}

	return estimated_contour_area;
}

unsigned int* ContourBuilder::get_estimated_contour() {
	return estimated_contour;
}

void ContourBuilder::generate_segments() throw (Exception) {
	unsigned int min_length = 0u;
	unsigned int residual_length = 0u;
	vector<unsigned int> lengths;

	min_length = db->get_n_markers() / n_segments;
	residual_length = db->get_n_markers() % n_segments;

	if (min_length == 1u) {
		throw Exception(__FILE__, __LINE__, "The number of segments is too high.");
	}

	for (unsigned int i = 0u; i < n_segments; ++i) {
		lengths.push_back(min_length);
	}

	for (unsigned int i = 0u; i < residual_length; ++i) {
		lengths.at(i) += 1u;
	}

	random_shuffle(lengths.begin(), lengths.end());

	segments = (Segment*)malloc(n_segments * sizeof(Segment));
	if (segments == NULL) {
		throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
	}

	segments[0u].start = 0u;
	for (unsigned int i = 0u; i < n_segments - 1u; ++i) {
		segments[i].end = segments[i].start + lengths.at(i) - 1u;
		segments[i + 1u].start = segments[i].end + 1u;
	}
	segments[n_segments - 1u].end = segments[n_segments - 1u].start + lengths.at(n_segments - 1u) - 1u;

	lengths.clear();
}

Cell* ContourBuilder::generate_next_cell() {
	Cell* cell = NULL;

	if (segment_j >= n_segments) {
		return NULL;
	}

	cell = new Cell(
			segment_i,
			segment_j,
			segments[segment_i].start,
			segments[segment_i].end,
			segments[segment_j].start,
			segments[segment_j].end
	);

	if (segment_j == n_segments - 1u) {
		segment_j = segment_j - segment_i + 1u;
		segment_i = 0u;
	} else {
		++segment_i;
		++segment_j;
	}

	return cell;
}

void ContourBuilder::add_profile(Profile* profile) {
	if (auxiliary::fcmp(profile->get_pi(), proportion, EPSILON) >= 0) {
		for (unsigned int i = profile->get_start(); i <= profile->get_end(); ++i) {
			estimated_contour[i] = min(estimated_contour[i], profile->get_start());
		}
		++n_candidate_profiles;
	}
}

void ContourBuilder::build() throw (Exception) {
	Cell* cell = NULL;
	unsigned long int cell_id = 0ul;
	unsigned int segment_i = 0u;
	unsigned int segment_j = 0u;

	Profile* profile = NULL;
	Profile* west = NULL;
	Profile* south_west = NULL;
	Profile* south = NULL;

	max_pool_size = 2 * n_segments - 1u;

	unordered_map<unsigned long int, Profile*> pool(max_pool_size);
	unordered_map<unsigned long int, Profile*>::iterator pool_it;

	Cell::set_db(db);
	Cell::set_method(ci_method);
	Cell::set_samples(samples);
	Cell::set_seed(seed);

	Profile::set_alpha(alpha);

	while ((cell = generate_next_cell()) != NULL) {
		cell->get_coordinates(&segment_i, &segment_j);

		n_computations += cell->sample();

		west = south_west = south = NULL;
		if (segment_i != segment_j) {
			cell_id = auxiliary::get_cantor_pairing_number(segment_i, segment_j - 1ul);
			pool_it = pool.find(cell_id);
			west = pool_it->second;

			cell_id = auxiliary::get_cantor_pairing_number(segment_i + 1ul, segment_j);
			pool_it = pool.find(cell_id);
			south = pool_it->second;

			if (segment_j - segment_i > 1) {
				cell_id = auxiliary::get_cantor_pairing_number(segment_i + 1ul, segment_j - 1ul);
				pool_it = pool.find(cell_id);
				south_west = pool_it->second;
				pool.erase(pool_it);
			}
		}

		profile = profile_factory->create(cell, west, south_west, south);

		pool.insert(pair<unsigned long int, Profile*>(auxiliary::get_cantor_pairing_number(segment_i, segment_j), profile));

		profile->estimate_ld_proportion();
		add_profile(profile);

		if (south_west != NULL) {
			delete south_west;
			south_west = NULL;
		}

		delete cell;
		cell = NULL;
	}

	pool_it = pool.begin();
	while (pool_it != pool.end()) {
		delete pool_it->second;
		++pool_it;
	}
}

double ContourBuilder::get_memory_usage() {
	double memory_usage = 0.0;

	if (auxiliary::strcmp_ignore_case(sci_method, ProfileFactory::FS) == 0) {
		memory_usage += (max_pool_size * sizeof(Profile_FS)) / 1048576.0;
	}

	memory_usage += (max_pool_size * (sizeof(unsigned long int) + sizeof(Profile*))) / 1048576.0;
	memory_usage += (n_segments * sizeof(Segment)) / 1048576.0;

	return memory_usage;
}
