/*
 * Copyright  2014 Daniel Taliun, Johann Gamper and Cristian Pattaro. All rights reserved.
 *
 * This file is part of S-MIG++.
 *
 * S-MIG++ is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * S-MIG++ is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with S-MIG++. If not, see <http://www.gnu.org/licenses/>.
 */

#include "include/MIG.h"

const unsigned int MIG::STRONG_PAIRS_SIZE_INIT = 100000;
const unsigned int MIG::STRONG_PAIRS_SIZE_INCREMENT = 1000;
const unsigned int MIG::BLOCKS_SIZE_INIT = 10000;
const unsigned int MIG::BLOCKS_SIZE_INCREMENT = 1000;

const double MIG::EPSILON = 0.000000001;

MIG::MIG(Db& db) throw (Exception) : db(&db),
		strong_pairs(NULL), n_strong_pairs(0u), strong_pairs_size(STRONG_PAIRS_SIZE_INIT),
		blocks(NULL), n_blocks(0u), blocks_size(BLOCKS_SIZE_INIT), w_values_random(NULL), w_values_random_n(NULL) {

	strong_pairs = (pair*)malloc(strong_pairs_size * sizeof(pair));
	if (strong_pairs == NULL) {
		throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
	}

	blocks = (unsigned int*)malloc(blocks_size * sizeof(unsigned int));
	if (blocks == NULL) {
		throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
	}

	w_values_random = (long double*)malloc(db.n_markers * sizeof(long double));
	if (w_values_random == NULL) {
		throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
	}

	w_values_random_n = (unsigned int*)malloc(db.n_markers * sizeof(unsigned int));
	if (w_values_random_n == NULL) {
		throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
	}
}

MIG::~MIG() {
	db = NULL;

	free(strong_pairs);
	strong_pairs = NULL;

	free(blocks);
	blocks = NULL;

	free(w_values_random);
	w_values_random = NULL;

	free(w_values_random_n);
	w_values_random_n = NULL;
}

int MIG::paircmp(const void* first, const void* second) {
	pair* first_pair = (pair*)first;
	pair* second_pair = (pair*)second;

	if (first_pair->distance > second_pair->distance) {
		return -1;
	} else if (first_pair->distance < second_pair->distance) {
		return 1;
	} else {
		return first_pair->first - second_pair->first;
	}
}

unsigned long int MIG::compute_candidate_blocks_migpp(unsigned int* contour, const char* ci_method, unsigned int threshold) throw (Exception) {
	AlgorithmCI* algorithm_ci = NULL;

	long double* w_values = NULL;
	long double* w_values_sums = NULL;

	long double w_values_sum_left = 0.0;
	long double* w_values_sums_left = NULL;

	long double w_value_max = 0.0;
	long double* w_values_max = NULL;

	long int* breakpoints = NULL;
	long int* terminations = NULL;

	interval* max_intervals = NULL;
	unsigned int n_max_intervals = 0u;

	double lower_ci = 0.0;
	double upper_ci = 0.0;

	unsigned int current_threshold = 0u;

	long int breakpoint = 0;
	long int updated_breakpoint = 0;

	unsigned long int calculations = numeric_limits<unsigned long int>::max();;
	unsigned long int total_calculations = 0ul;

	pair* new_strong_pairs = NULL;

	algorithm_ci = AlgorithmFactory::create(*db, ci_method);

	w_values = (long double*)malloc(db->n_markers * sizeof(long double));
	if (w_values == NULL) {
		throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
	}

	w_values_sums = (long double*)malloc(db->n_markers * sizeof(long double));
	if (w_values_sums == NULL) {
		throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
	}

	w_values_sums_left = (long double*)malloc(db->n_markers * sizeof(long double));
	if (w_values_sums_left == NULL) {
		throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
	}

	w_values_max = (long double*)malloc(db->n_markers * sizeof(long double));
	if (w_values_max == NULL) {
		throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
	}

	terminations = (long int*)malloc(db->n_markers * sizeof(long int));
	if (terminations == NULL) {
		throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
	}

	breakpoints = (long int*)malloc(db->n_markers * sizeof(long int));
	if (breakpoints == NULL) {
		throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
	}

	for (unsigned int i = 0u; i < db->n_markers; ++i) {
		w_values[i] = 0.0;
		w_values_sums[i] = 0.0;
		terminations[i] = i;
		breakpoints[i] = i;
	}

	for (unsigned int i = 0u; i < db->n_markers; ++i) {
		w_values_sum_left += 0.05 * (terminations[i] - contour[i]);
		w_values_sums_left[i] = w_values_sum_left;
	}

	for (unsigned int i = 0u, max_start = 0u; i < db->n_markers; ++i) {
		if (max_start != contour[i]) {
			max_start = contour[i];
			++n_max_intervals;
		}
	}
	++n_max_intervals;

	max_intervals = (interval*)malloc(n_max_intervals * sizeof(interval));
	if (max_intervals == NULL) {
		throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
	}

	max_intervals[0u].start = contour[0u];
	for (unsigned int i = 1u, j = 0u; i < db->n_markers; ++i) {
		if (max_intervals[j].start != contour[i]) {
			max_intervals[j].end = i - 1u;
			max_intervals[++j].start = contour[i];
		}
	}
	max_intervals[n_max_intervals - 1u].end = db->n_markers - 1u;

	for (unsigned int i = 0u; i < n_max_intervals; ++i) {
		w_values_max[max_intervals[i].end] = w_values_sums_left[max_intervals[i].end];
		for (long int j = max_intervals[i].end - 1u; j >= max_intervals[i].start; --j) {
			w_values_max[j] = w_values_max[j + 1u] > w_values_sums_left[j] ? w_values_max[j + 1u] : w_values_sums_left[j];
		}
	}

	while (calculations > 0u) {
		current_threshold += threshold;

		calculations = 0u;

		breakpoint = 0;
		updated_breakpoint = 0;

		w_values_sum_left = 0.0;

		for (long int i = 1; i < db->n_markers; ++i) {
			if (updated_breakpoint < contour[i]) {
				updated_breakpoint = contour[i];
			}

			if (updated_breakpoint == breakpoints[i]) {
				breakpoints[i] = breakpoint;
				breakpoint = updated_breakpoint = terminations[i];

				w_values_sum_left += 0.05 * (terminations[i] - contour[i]) + w_values_sums[i];
				w_values_sums_left[i] = w_values_sum_left;

				continue;
			}

			if ((i - updated_breakpoint) > current_threshold) {
				breakpoints[i] = breakpoint = i - current_threshold;
			} else {
				breakpoints[i] = breakpoint;
				breakpoint = updated_breakpoint;
			}

			updated_breakpoint = terminations[i];

			for (long int j = terminations[i] - 1u; j >= breakpoint; --j) {
				++calculations;

				algorithm_ci->get_CI(i, j, &lower_ci, &upper_ci);
				if (!std::isnan(lower_ci) && !std::isnan(upper_ci)) {
					if (((auxiliary::fcmp(lower_ci, 0.7, EPSILON) >= 0) && (auxiliary::fcmp(upper_ci, 0.98, EPSILON) >= 0)) ||
							((auxiliary::fcmp(lower_ci, -0.98, EPSILON) <= 0) && (auxiliary::fcmp(upper_ci, -0.7, EPSILON) <= 0))) {
						w_values_sums[i] += 0.05;
						w_values[j] += w_values_sums[i];
						if (auxiliary::fcmp(w_values[j], 0.0, EPSILON) >= 0) {
							if (n_strong_pairs >= strong_pairs_size) {
								strong_pairs_size += STRONG_PAIRS_SIZE_INCREMENT;
								new_strong_pairs = (pair*)realloc(strong_pairs, strong_pairs_size * sizeof(pair));
								if (new_strong_pairs == NULL) {
									throw Exception(__FILE__, __LINE__, "Error in memory reallocation.");
								}
								strong_pairs = new_strong_pairs;
								new_strong_pairs = NULL;
							}

							strong_pairs[n_strong_pairs].first = j;
							strong_pairs[n_strong_pairs].last = i;
							strong_pairs[n_strong_pairs].distance = db->positions[i] - db->positions[j];

							++n_strong_pairs;
						}
					} else if ((auxiliary::fcmp(lower_ci, -0.9, EPSILON) >= 0) && (auxiliary::fcmp(upper_ci, 0.9, EPSILON) <= 0)) {
						w_values_sums[i] -= 0.95;
						w_values[j] += w_values_sums[i];
					} else {
						w_values[j] += w_values_sums[i];
					}
				} else {
					w_values[j] += w_values_sums[i];
				}

				/* 004: with prior using pre-calculated sums medium conservative */
				w_value_max = w_values[j] + w_values_max[i] - w_values_sums_left[i];
				if (auxiliary::fcmp(w_value_max, 0.0, EPSILON) >= 0) {
					updated_breakpoint = j;
				}
			}

			terminations[i] = breakpoint;

			w_values_sum_left += 0.05 * (terminations[i] - contour[i]) + w_values_sums[i];
			w_values_sums_left[i] = w_values_sum_left;
		}

		for (unsigned int i = 0u; i < n_max_intervals; ++i) {
			w_values_max[max_intervals[i].end] = w_values_sums_left[max_intervals[i].end];
			for (long int j = max_intervals[i].end - 1u; j >= max_intervals[i].start; --j) {
				w_values_max[j] = w_values_max[j + 1u] > w_values_sums_left[j] ? w_values_max[j + 1u] : w_values_sums_left[j];
			}
		}

		total_calculations += calculations;
	}

	delete algorithm_ci;
	algorithm_ci = NULL;

	free(w_values);
	w_values = NULL;

	free(w_values_sums);
	w_values_sums = NULL;

	free(w_values_sums_left);
	w_values_sums_left = NULL;

	free(w_values_max);
	w_values_max = NULL;

	free(terminations);
	terminations = NULL;

	free(breakpoints);
	breakpoints = NULL;

	free(max_intervals);
	max_intervals = NULL;

	return total_calculations;
}

void MIG::sort_candidate_blocks() {
	qsort(strong_pairs, n_strong_pairs, sizeof(pair), paircmp);
}

void MIG::get_contour(unsigned int** contour, unsigned long int* contour_area) throw (Exception) {
	*contour = (unsigned int*)malloc(db->n_markers * sizeof(unsigned int));
	if (*contour == NULL) {
		throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
	}

	for (unsigned int i = 0u; i < db->n_markers; ++i) {
		(*contour)[i] = i;
	}

	*contour_area = 0ul;

	for (unsigned int p = 0u; p < n_strong_pairs; ++p) {
		for (unsigned int i = strong_pairs[p].first; i < strong_pairs[p].last; ++i) {
			(*contour)[i] = min((*contour)[i], strong_pairs[p].first);
		}
	}

	for (unsigned int i = 1u; i < db->n_markers; ++i) {
		*contour_area += i - (*contour)[i];
	}
}

void MIG::write_contour(unsigned int* contour, const char* output_file_name, const char* output_type) throw (Exception) {
	Writer* writer = WriterFactory::create(output_type);
	writer->set_file_name(output_file_name);
	writer->open();

	writer->write("ID1\tID2\n");

	for (unsigned int i = 0u; i < db->n_markers; ++i) {
		writer->write("%u\t%u\n", i, contour[i]);
	}

	writer->close();
	delete writer;
	writer = NULL;
}

void MIG::select_final_blocks() throw (Exception) {
	unsigned int* new_blocks = NULL;
	bool* used_markers = NULL;

	unsigned int first = 0u;
	unsigned int last = 0u;

	n_blocks = 0u;

	used_markers = (bool*)malloc(db->n_markers * sizeof(bool));
	if (used_markers == NULL) {
		throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
	}

	for (unsigned int i = 0u; i < db->n_markers; ++i) {
		used_markers[i] = false;
	}

	for (unsigned int b = 0u; b < n_strong_pairs; ++b) {
		first = strong_pairs[b].first;
		last = strong_pairs[b].last;

		if (used_markers[first] || used_markers[last]) {
			continue;
		}

		if (n_blocks >= blocks_size) {
			blocks_size += BLOCKS_SIZE_INCREMENT;
			new_blocks = (unsigned int*)realloc(blocks, blocks_size * sizeof(unsigned int));
			if (new_blocks == NULL) {
				throw Exception(__FILE__, __LINE__, "Error in memory reallocation.");
			}
			blocks = new_blocks;
			new_blocks = NULL;
		}

		blocks[n_blocks] = b;
		++n_blocks;

		for (unsigned int i = first; i <= last; ++i) {
			used_markers[i] = true;
		}
	}

	free(used_markers);
	used_markers = NULL;
}

unsigned int MIG::get_n_strong_pairs() {
	return n_strong_pairs;
}

unsigned int MIG::get_n_blocks() {
	return n_blocks;
}

void MIG::write_blocks(const char* output_file_name,  const char* output_type,
		const char* input_phase_file_name, const char* input_map_file_name,
		double maf_threshold, bool region, unsigned long int start, unsigned long int end,
		const char* ci_method) throw (Exception) {
	Writer* writer = NULL;

	pair strong_pair;

	const char* first_marker = NULL;
	const char* last_marker = NULL;
	unsigned int start_bp = 0u;
	unsigned int end_bp = 0u;
	unsigned int n_markers = 0u;
	unsigned int n_haps = 0u;
	unsigned int n_unique_haps = 0u;
	unsigned int n_common_haps = 0u;
	double haps_diversity = 0.0;

	try {
		writer = WriterFactory::create(output_type);
		writer->set_file_name(output_file_name);
		writer->open();

		writer->write("#PHASE FILE: %s\n", input_phase_file_name);
		if (input_map_file_name != NULL) {
			writer->write("#MAP FILE: %s\n", input_map_file_name);
		}
		if (region) {
			writer->write("#REGION: [%u, %u]\n", start, end);
		}
		writer->write("#HAPLOTYPES: %u\n", db->n_haplotypes);
		writer->write("#MARKERS: %u\n", db->all_n_markers);
		writer->write("#METHOD: %s\n", ci_method);
		writer->write("#MAF > %g\n", maf_threshold);
		writer->write("#MARKERS (MAF > %g): %u\n", maf_threshold, db->n_markers);

		writer->write("%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\t%s\n",
				"BLOCK_NAME", "FIRST_MARKER", "LAST_MARKER", "FIRST_MARKER_ID", "LAST_MARKER_ID", "START_BP", "END_BP", "N_MARKERS", "N_HAPS", "N_UNIQUE_HAPS", "N_COMMON_HAPS", "HAPS_DIVERSITY");

		for (unsigned int b = 0u; b < n_blocks; ++b) {
			strong_pair = strong_pairs[blocks[b]];

			first_marker = db->markers[strong_pair.first];
			last_marker = db->markers[strong_pair.last];
			start_bp = db->positions[strong_pair.first];
			end_bp = db->positions[strong_pair.last];
			n_markers = strong_pair.last - strong_pair.first + 1u;

			get_block_diversity(b, &n_haps, &n_unique_haps, &n_common_haps, &haps_diversity);

			writer->write("BLOCK_%07u\t%s\t%s\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%g\n",
					b + 1u, first_marker, last_marker, strong_pair.first, strong_pair.last, start_bp, end_bp, n_markers, n_haps, n_unique_haps, n_common_haps, haps_diversity);
		}

		writer->close();
		delete writer;
	} catch (Exception &e) {
		if (writer != NULL) {
			delete writer;
		}
		throw;
	}
}

void MIG::write_candidate_blocks(const char* output_file_name, const char* output_type) throw (Exception) {
	Writer* writer = NULL;

	pair strong_pair;

	try {
		writer = WriterFactory::create(output_type);
		writer->set_file_name(output_file_name);
		writer->open();

		writer->write("FIRST_MARKER_ID\tLAST_MARKER_ID\n");
		for (unsigned int pb = 0u; pb < n_strong_pairs; ++pb) {
			strong_pair = strong_pairs[pb];

			writer->write("%u\t%u\n", strong_pair.first, strong_pair.last);
		}

		writer->close();
		delete writer;
	} catch (Exception &e) {
		if (writer != NULL) {
			delete writer;
		}
		throw;
	}
}

bool MIG::is_compatible_haplotype(const char* first, const char* second) {
	unsigned int i = 0u;
	unsigned int length = max(strlen(first), strlen(second));
	char first_char = '\0';
	char second_char = '\0';

	while (i < length) {
		first_char = tolower(first[i]);
		if ((first_char != 'a') && (first_char != 'c') && (first_char != 'g') && (first_char != 't')) {
			++i;
			continue;
		}

		second_char = tolower(second[i]);
		if ((second_char != 'a') && (second_char != 'c') && (second_char != 'g') && (second_char != 't')) {
			++i;
			continue;
		}

		if (first_char != second_char) {
			return false;
		}

		++i;
	}

	if ((first[i] == '\0') && (second[i] == '\0')) {
		return true;
	}

	return false;
}

void MIG::get_block_diversity(unsigned int block_id, unsigned int* n_haps, unsigned int* n_unique_haps, unsigned int* n_common_haps, double* haps_diversity) throw (Exception) {
	pair strong_pair;

	unsigned int n_markers = 0u;
	unsigned int n_all_common_haps = 0u;

	char* hap = NULL;
	char* hap_new = NULL;

	map<char*, unsigned int, bool(*)(const char*, const char*)> haps(auxiliary::bool_strcmp_ignore_case);
	map<char*, unsigned int, bool(*)(const char*, const char*)>::iterator haps_it;

	vector<const char*> compatible_haps;

	map<const char*, unsigned int, bool(*)(const char*, const char*)> unambiguous_haps(auxiliary::bool_strcmp_ignore_case);
	map<const char*, unsigned int, bool(*)(const char*, const char*)>::iterator unambiguous_haps_it;

	vector< std::pair<const char*, unsigned int> > unique_haps;
	vector< std::pair<const char*, unsigned int> >::iterator unique_haps_it;

	try {
		*n_haps = 0u;
		*n_unique_haps = 0u;
		*n_common_haps = 0u;
		*haps_diversity = 0.0;

		strong_pair = strong_pairs[blocks[block_id]];

		n_markers = strong_pair.last - strong_pair.first + 1u;

		hap = (char*)malloc((n_markers + 1u) * sizeof(char));
		if (hap == NULL) {
			throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
		}
		hap[n_markers] = '\0';

		/* Read all (ambiguous & unambiguous) haplotypes */
		for (unsigned int j = 0u; j < db->n_haplotypes; ++j) {
			for (unsigned int i = strong_pair.first, k = 0u; i <= strong_pair.last; ++i, ++k) {
				hap[k] = db->haplotypes[i][j];
			}

			haps_it = haps.find(hap);
			if (haps_it == haps.end()) {
				hap_new = (char*)malloc((n_markers + 1u) * sizeof(char));
				if (hap_new == NULL) {
					throw Exception(__FILE__, __LINE__, "Error in memory allocation.");
				}
				strcpy(hap_new, hap);

				haps.insert(std::pair<char*, unsigned int>(hap_new, 1u));
			} else {
				haps_it->second++;
			}
		}

		/* Select unambiguous haplotypes */
		haps_it = haps.begin();
		while (haps_it != haps.end()) {
			unambiguous_haps.insert(std::pair<const char*, unsigned int>(haps_it->first, haps_it->second));
			++haps_it;
		}

		haps_it = haps.begin();
		while (haps_it != haps.end()) {
			compatible_haps.clear();

			unambiguous_haps_it = unambiguous_haps.begin();
			while (unambiguous_haps_it != unambiguous_haps.end()) {
				if (is_compatible_haplotype(haps_it->first, unambiguous_haps_it->first)) {
					compatible_haps.push_back(unambiguous_haps_it->first);
				}
				++unambiguous_haps_it;
			}

			for (unsigned int j = 1u; j < compatible_haps.size(); ++j) {
				for (unsigned int i = 0u; i < j; ++i) {
					if (!is_compatible_haplotype(compatible_haps.at(i), compatible_haps.at(j))) {
						unambiguous_haps.erase(haps_it->first);
						break;
					}
				}
			}

			++haps_it;
		}

		/* Group unambiguous haplotypes by compatibility */
		unambiguous_haps_it = unambiguous_haps.begin();
		while (unambiguous_haps_it != unambiguous_haps.end()) {

			unique_haps_it = unique_haps.begin();
			while (unique_haps_it != unique_haps.end()) {
				if (is_compatible_haplotype(unambiguous_haps_it->first, unique_haps_it->first)) {
					unique_haps_it->second += unambiguous_haps_it->second;
					break;
				}
				++unique_haps_it;
			}

			if (unique_haps_it == unique_haps.end()) {
				unique_haps.push_back(std::pair<const char*, unsigned int>(unambiguous_haps_it->first, unambiguous_haps_it->second));
			}

			++unambiguous_haps_it;
		}

		unique_haps_it = unique_haps.begin();
		while (unique_haps_it != unique_haps.end()) {
			*n_haps += unique_haps_it->second;
			if (unique_haps_it->second > 1u) {
				*n_common_haps += 1u;
				n_all_common_haps += unique_haps_it->second;
			}
			++unique_haps_it;
		}

		*n_unique_haps = unique_haps.size();
		*haps_diversity = ((double)n_all_common_haps) / ((double)*n_haps);

		compatible_haps.clear();
		unambiguous_haps.clear();
		unique_haps.clear();

		for (haps_it = haps.begin(); haps_it != haps.end(); ++haps_it) {
			free(haps_it->first);
		}
		haps.clear();

		free(hap);
	} catch (Exception &e) {
		compatible_haps.clear();
		unambiguous_haps.clear();
		unique_haps.clear();

		for (haps_it = haps.begin(); haps_it != haps.end(); ++haps_it) {
			free(haps_it->first);
		}
		haps.clear();

		if (hap != NULL) {
			free(hap);
		}

		throw;
	}
}

long double MIG::get_proportion(Db* db, const char* ci_method, unsigned int start_i, unsigned int end_i, unsigned int start_j, unsigned int end_j) throw (Exception) {
	AlgorithmCI* algorithm_ci = NULL;

	double lower_ci = 0.0;
	double upper_ci = 0.0;

	unsigned int n_strong_pairs = 0u;
	unsigned int n_recomb_pairs = 0u;
	unsigned int n_computations = 0u;

	algorithm_ci = AlgorithmFactory::create(*db, ci_method);

	for (unsigned int i = start_i; i < end_i; ++i) {
		for (unsigned int j = start_j; j < end_j; ++j) {
			++n_computations;
			algorithm_ci->get_CI(i, j, &lower_ci, &upper_ci);
			if (!std::isnan(lower_ci) && !std::isnan(upper_ci)) {
				if (((auxiliary::fcmp(lower_ci, 0.7, EPSILON) >= 0) && (auxiliary::fcmp(upper_ci, 0.98, EPSILON) >= 0)) ||
						((auxiliary::fcmp(lower_ci, -0.98, EPSILON) <= 0) && (auxiliary::fcmp(upper_ci, -0.7, EPSILON) <= 0))) {
					++n_strong_pairs;
				} else if ((auxiliary::fcmp(lower_ci, -0.9, EPSILON) >= 0) && (auxiliary::fcmp(upper_ci, 0.9, EPSILON) <= 0)) {
					++n_recomb_pairs;
				}
			}
		}
	}

	return ((long double)n_strong_pairs) / ((long double)(n_recomb_pairs + n_strong_pairs));
}

double MIG::get_max_memory_usage() {
	double memory_usage = 0.0;

	memory_usage += (4u * db->n_markers * sizeof(long double)) / 1048576.0;
	memory_usage += (2u * db->n_markers * sizeof(long int)) / 1048576.0;
	memory_usage += (db->n_markers * sizeof(interval)) / 1048576.0;
	memory_usage += (db->n_markers * sizeof(unsigned int)) / 1048576.0;
	memory_usage += (strong_pairs_size * sizeof(pair)) / 1048576.0;
	memory_usage += (blocks_size * sizeof(unsigned int)) / 1048576.0;

	return memory_usage;
}
