/*
 * Copyright  2014 Daniel Taliun, Johann Gamper and Cristian Pattaro. All rights reserved.
 *
 * This file is part of S-MIG++.
 *
 * S-MIG++ is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * S-MIG++ is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with S-MIG++. If not, see <http://www.gnu.org/licenses/>.
 */

#include "include/Profile.h"

const double Profile::EPSILON = 0.000000001;

double Profile::alpha = numeric_limits<double>::quiet_NaN();
double Profile::chisq_2df = numeric_limits<double>::quiet_NaN();
double Profile::z_gaussian = numeric_limits<double>::quiet_NaN();

Profile::Profile(Cell* cell, Profile* west, Profile* south_west, Profile* south) :
	n_references(0), segment_i(0u), segment_j(0u), start(0u), end(0u),
	n_total(0ul), n_sampled_noninf(0ul), n_sampled_ld(0ul), n_sampled_ehr(0ul),
	min_n_ehr(numeric_limits<double>::quiet_NaN()),
	upper_ld(numeric_limits<double>::quiet_NaN()), lower_ehr(numeric_limits<double>::quiet_NaN()), pi(numeric_limits<double>::quiet_NaN()) {

	cell->get_coordinates(&segment_i, &segment_j);

	start = cell->get_start_i();
	end = cell->get_end_j();

	n_total += cell->get_n_total();
	n_sampled_noninf += cell->get_n_sampled_noninf();
	n_sampled_ld += cell->get_n_sampled_ld();
	n_sampled_ehr += cell->get_n_sampled_ehr();

	if (west != NULL) {
		n_total += west->n_total;
		n_sampled_noninf += west->n_sampled_noninf;
		n_sampled_ld += west->n_sampled_ld;
		n_sampled_ehr += west->n_sampled_ehr;
	}

	if (south != NULL) {
		n_total += south->n_total;
		n_sampled_noninf += south->n_sampled_noninf;
		n_sampled_ld += south->n_sampled_ld;
		n_sampled_ehr += south->n_sampled_ehr;
	}

	if (south_west != NULL) {
		n_total -= south_west->n_total;
		n_sampled_noninf -= south_west->n_sampled_noninf;
		n_sampled_ld -= south_west->n_sampled_ld;
		n_sampled_ehr -= south_west->n_sampled_ehr;

		min_n_ehr = south_west->lower_ehr * south_west->n_total;
	}
}

Profile::~Profile() {

}

void Profile::set_alpha(double alpha) throw (Exception) {
	Profile::alpha = alpha;

	Profile::chisq_2df = gsl_cdf_chisq_Pinv(1.0 - Profile::alpha, 2.0);
	if (gsl_isnan(Profile::chisq_2df) == 1) {
		throw Exception(__FILE__, __LINE__, "ChiSq CDF value is NaN.");
	}
	if (gsl_isinf(Profile::chisq_2df) != 0) {
		throw Exception(__FILE__, __LINE__, "ChiSq CDF value is infinite.");
	}

	Profile::z_gaussian = abs(gsl_cdf_gaussian_Pinv(1.0 - Profile::alpha / 2.0, 1.0));
	if (gsl_isnan(Profile::z_gaussian) == 1) {
		throw Exception(__FILE__, __LINE__, "Gaussian CDF value is NaN.");
	}
	if (gsl_isinf(Profile::z_gaussian) != 0) {
		throw Exception(__FILE__, __LINE__, "Gaussian CDF value is infinite.");
	}
}

void Profile::set_n_references(int n_references) {
	this->n_references = n_references;
}

void Profile::remove_reference() {
	--n_references;
}

int Profile::get_n_references() {
	return n_references;
}

unsigned int Profile::get_start() {
	return start;
}

unsigned int Profile::get_end() {
	return end;
}

double Profile::get_pi() {
	return pi;
}

void Profile::estimate_ld_proportion() {
	estimate_simultaneous_ci();

	if (!std::isnan(min_n_ehr)) {
		pi = (upper_ld * n_total) / (upper_ld * n_total + min_n_ehr);
	} else {
		pi = 1.0;
	}
}

void Profile::write(Writer* writer) throw (Exception) {
	writer->write("%u\t%u\t%u\t%u\t%u\t%u\t%u\t%u\t%g\t%g\t%g\n",
			segment_i, segment_j, start, end, n_total, n_sampled_ld, n_sampled_ehr, n_sampled_noninf, upper_ld, lower_ehr, pi);
}
